<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";
csrf_token();

$load_id = (int)($_GET['load_id'] ?? 0);
if (!$load_id) {
    flash_set('error', 'Missing loading id.');
    redirect('/smartbake360/modules/dispatch/loadings_list.php');
}

// Load dispatch header
$stmt = $conn->prepare("
  SELECT dl.*, v.reg_no, r.name AS rider_name
  FROM dispatch_loads dl
  LEFT JOIN vehicles v ON dl.vehicle_id = v.id
  LEFT JOIN riders r ON dl.rider_id = r.id
  WHERE dl.id = ?
");
$stmt->bind_param("i", $load_id);
$stmt->execute();
$load = $stmt->get_result()->fetch_assoc();

if (!$load) {
    flash_set('error', 'Loading not found.');
    redirect('/smartbake360/modules/dispatch/loadings_list.php');
}

$page_title = "Route Cash Reconciliation — " . $load['load_no'];

// 1) Loaded / returned / sold (system) with value
$sql = "
  SELECT dli.product_id,
         SUM(dli.qty_loaded)   AS qty_loaded,
         SUM(dli.qty_returned) AS qty_returned,
         p.code,
         p.name,
         p.selling_price
  FROM dispatch_load_items dli
  JOIN products p ON dli.product_id = p.id
  WHERE dli.load_id = ?
  GROUP BY dli.product_id, p.code, p.name, p.selling_price
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $load_id);
$stmt->execute();
$items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$loaded_pcs = 0;
$loaded_val = 0.0;
$returned_pcs = 0;
$returned_val = 0.0;

foreach ($items as $it) {
    $loaded_pcs   += (int)$it['qty_loaded'];
    $return_pcs   = (int)$it['qty_returned'];
    $returned_pcs += $return_pcs;

    $price = (float)$it['selling_price'];
    $loaded_val   += (int)$it['qty_loaded']   * $price;
    $returned_val += $return_pcs * $price;
}

$sold_sys_pcs = $loaded_pcs - $returned_pcs;
$sold_sys_val = $loaded_val - $returned_val;

// 2) Van sales totals for this load
$sql = "
  SELECT 
    SUM(vsi.qty)         AS sold_qty,
    SUM(vsi.line_total)  AS sold_value,
    SUM(vs.cash_amount)  AS cash_total,
    SUM(vs.mpesa_amount) AS mpesa_total,
    SUM(vs.credit_amount)AS credit_total
  FROM van_sales_items vsi
  JOIN van_sales vs ON vsi.van_sale_id = vs.id
  WHERE vs.load_id = ?
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $load_id);
$stmt->execute();
$vs = $stmt->get_result()->fetch_assoc();

$sold_van_pcs   = (int)($vs['sold_qty'] ?? 0);
$sold_van_val   = (float)($vs['sold_value'] ?? 0);
$expected_cash  = (float)($vs['cash_total'] ?? 0);
$expected_mpesa = (float)($vs['mpesa_total'] ?? 0);
$expected_credit= (float)($vs['credit_total'] ?? 0);
$expected_total = $expected_cash + $expected_mpesa + $expected_credit;

// 3) Existing cash drop (if any)
$stmt = $conn->prepare("
  SELECT * FROM route_cash_drops WHERE load_id = ? ORDER BY id DESC LIMIT 1
");
$stmt->bind_param("i", $load_id);
$stmt->execute();
$drop = $stmt->get_result()->fetch_assoc();

$actual_cash   = $drop['actual_cash']   ?? 0;
$actual_mpesa  = $drop['actual_mpesa']  ?? 0;
$actual_total  = $drop['actual_total']  ?? 0;
$drop_remarks  = $drop['remarks']       ?? '';
$drop_date_val = $drop['drop_date']     ?? date('Y-m-d H:i:s');

// Differences
$qty_diff_sys_vs_van = $sold_sys_pcs - $sold_van_pcs;
$val_diff_sys_vs_van = $sold_sys_val - $sold_van_val;

$cash_diff   = $actual_cash  - $expected_cash;
$mpesa_diff  = $actual_mpesa - $expected_mpesa;
$grand_exp   = $expected_total;
$grand_act   = $actual_total;
$grand_diff  = $grand_act - $grand_exp;
?>
<?php require_once __DIR__ . "/../../includes/layout.php"; ?>
<div class="sb-card mb-3">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <h5 class="mb-0">Route Cash Reconciliation</h5>
      <p class="small text-muted mb-0">
        Load <?php echo h($load['load_no']); ?> • Date: <?php echo h($load['load_date']); ?> • Route: <?php echo h($load['route_code']); ?><br>
        Vehicle: <?php echo h($load['reg_no'] ?: '-'); ?> • Rider: <?php echo h($load['rider_name'] ?: '-'); ?>
      </p>
    </div>
    <a href="/smartbake360/modules/dispatch/loadings_list.php"
       class="btn btn-sm btn-outline-light rounded-pill">
      <i class="fa-solid fa-arrow-left-long me-1"></i> Back
    </a>
  </div>

  <div class="row g-3 small">
    <div class="col-md-4">
      <div class="border rounded-4 p-3 bg-dark bg-opacity-50 h-100">
        <h6 class="text-uppercase text-muted small mb-2">Stock Movement</h6>
        <p class="mb-1">
          Loaded: <strong><?php echo number_format($loaded_pcs); ?> pcs</strong><br>
          Value Loaded: <strong>KES <?php echo number_format($loaded_val, 2); ?></strong>
        </p>
        <p class="mb-1">
          Returned: <strong><?php echo number_format($returned_pcs); ?> pcs</strong><br>
          Value Returned: <strong>KES <?php echo number_format($returned_val, 2); ?></strong>
        </p>
        <p class="mb-0">
          Sold (System): <strong><?php echo number_format($sold_sys_pcs); ?> pcs</strong><br>
          Value Sold (System): <strong>KES <?php echo number_format($sold_sys_val, 2); ?></strong>
        </p>
      </div>
    </div>

    <div class="col-md-4">
      <div class="border rounded-4 p-3 bg-dark bg-opacity-50 h-100">
        <h6 class="text-uppercase text-muted small mb-2">Van Sales Summary</h6>
        <p class="mb-1">
          Sold (Van Sales): <strong><?php echo number_format($sold_van_pcs); ?> pcs</strong><br>
          Value Sold (Van Sales): <strong>KES <?php echo number_format($sold_van_val, 2); ?></strong>
        </p>
        <p class="mb-1">
          System vs Van (Qty): 
          <strong class="<?php echo $qty_diff_sys_vs_van == 0 ? 'text-success' : 'text-warning'; ?>">
            <?php echo number_format($qty_diff_sys_vs_van); ?> pcs
          </strong>
        </p>
        <p class="mb-0">
          System vs Van (Value): 
          <strong class="<?php echo abs($val_diff_sys_vs_van) < 1 ? 'text-success' : 'text-warning'; ?>">
            KES <?php echo number_format($val_diff_sys_vs_van, 2); ?>
          </strong>
        </p>
      </div>
    </div>

    <div class="col-md-4">
      <div class="border rounded-4 p-3 bg-dark bg-opacity-50 h-100">
        <h6 class="text-uppercase text-muted small mb-2">Expected vs Actual Money</h6>
        <p class="mb-1">
          Expected Cash: <strong>KES <?php echo number_format($expected_cash, 2); ?></strong><br>
          Expected Mpesa: <strong>KES <?php echo number_format($expected_mpesa, 2); ?></strong><br>
          Expected Credit: <strong>KES <?php echo number_format($expected_credit, 2); ?></strong><br>
          Expected Total: <strong>KES <?php echo number_format($expected_total, 2); ?></strong>
        </p>
        <p class="mb-1">
          Actual Cash: <strong>KES <?php echo number_format($actual_cash, 2); ?></strong><br>
          Actual Mpesa: <strong>KES <?php echo number_format($actual_mpesa, 2); ?></strong><br>
          Actual Total: <strong>KES <?php echo number_format($actual_total, 2); ?></strong>
        </p>
        <p class="mb-0">
          Overall Difference: 
          <strong class="<?php echo abs($grand_diff) < 1 ? 'text-success' : 'text-danger'; ?>">
            KES <?php echo number_format($grand_diff, 2); ?>
          </strong>
        </p>
      </div>
    </div>
  </div>

  <hr class="border-secondary my-3">

  <form method="post" action="route_cash_save.php" class="small">
    <input type="hidden" name="load_id" value="<?php echo (int)$load_id; ?>">
    <input type="hidden" name="csrf_token" value="<?php echo h($_SESSION['csrf_token']); ?>">
    <input type="hidden" name="expected_cash" value="<?php echo h($expected_cash); ?>">
    <input type="hidden" name="expected_mpesa" value="<?php echo h($expected_mpesa); ?>">
    <input type="hidden" name="expected_credit" value="<?php echo h($expected_credit); ?>">
    <input type="hidden" name="expected_total" value="<?php echo h($expected_total); ?>">

    <div class="row g-2 align-items-end">
      <div class="col-md-3">
        <label class="form-label">Drop Date &amp; Time</label>
        <input type="text" name="drop_date" class="form-control form-control-sm"
               value="<?php echo h($drop_date_val); ?>">
      </div>
      <div class="col-md-3">
        <label class="form-label">Actual Cash Dropped</label>
        <input type="number" step="0.01" name="actual_cash" class="form-control form-control-sm"
               value="<?php echo h($actual_cash); ?>">
      </div>
      <div class="col-md-3">
        <label class="form-label">Actual Mpesa (as per statement)</label>
        <input type="number" step="0.01" name="actual_mpesa" class="form-control form-control-sm"
               value="<?php echo h($actual_mpesa); ?>">
      </div>
      <div class="col-md-3">
        <label class="form-label">Remarks</label>
        <input type="text" name="remarks" class="form-control form-control-sm"
               value="<?php echo h($drop_remarks); ?>">
      </div>
    </div>

    <div class="d-flex justify-content-between align-items-center mt-3">
      <div class="small text-muted">
        Differences highlight possible <strong>short / over / Mpesa variance</strong>.
      </div>
      <button type="submit" class="btn btn-sm btn-primary rounded-pill"
              style="background:var(--sb-primary);border-color:var(--sb-primary-dark);">
        <i class="fa-solid fa-floppy-disk me-1"></i> Save Route Cash
      </button>
    </div>
  </form>
</div>
<?php require_once __DIR__ . "/../../includes/footer.php"; ?>
