<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";

$page_title = "Product Costing (Per Piece)";

$sql = "
  SELECT 
    r.id AS recipe_id,
    r.code AS recipe_code,
    r.name AS recipe_name,
    r.yield_qty,
    p.id AS product_id,
    p.code AS product_code,
    p.name AS product_name,
    p.category,
    p.selling_price
  FROM recipes r
  JOIN products p ON r.product_id = p.id
  ORDER BY p.category, p.name
";
$recipes = $conn->query($sql);

function compute_cost($conn, $recipe_id) {
    $sql = "
      SELECT 
        ri.qty,
        rm.unit_cost
      FROM recipe_items ri
      JOIN raw_materials rm ON ri.material_id = rm.id
      WHERE ri.recipe_id = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $recipe_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $total = 0;
    while($row = $res->fetch_assoc()) {
        $total += (float)$row['qty'] * (float)$row['unit_cost'];
    }
    return $total;
}
?>
<?php require_once __DIR__ . "/../../includes/layout.php"; ?>
<div class="sb-card mb-3">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <h5 class="mb-0">Product Costing</h5>
      <p class="small text-muted mb-0">
        Cost per piece from recipe (ingredients x unit cost) vs selling price and margin.
      </p>
    </div>
    <a href="recipes_list.php" class="btn btn-sm btn-outline-light rounded-pill">
      <i class="fa-solid fa-list-check me-1"></i> Manage Recipes
    </a>
  </div>
  <div class="table-responsive small">
    <table class="table table-dark table-borderless align-middle mb-0">
      <thead class="text-muted text-uppercase small">
        <tr>
          <th>Product</th>
          <th>Recipe</th>
          <th class="text-end">Yield (pcs)</th>
          <th class="text-end">Batch Cost (KES)</th>
          <th class="text-end">Cost / Piece (KES)</th>
          <th class="text-end">Selling Price</th>
          <th class="text-end">Margin (KES)</th>
          <th class="text-end">Margin %</th>
        </tr>
      </thead>
      <tbody>
        <?php while($r = $recipes->fetch_assoc()): ?>
          <?php
            $batch_cost = compute_cost($conn, $r['recipe_id']);
            $yield      = max(1, (int)$r['yield_qty']);
            $cost_piece = $batch_cost / $yield;
            $sp         = (float)$r['selling_price'];
            $margin     = $sp - $cost_piece;
            $margin_pct = $sp > 0 ? ($margin / $sp) * 100 : 0;
          ?>
          <tr>
            <td>
              <strong><?php echo h($r['product_code']); ?></strong><br>
              <span class="text-muted"><?php echo h($r['product_name']); ?></span>
              <span class="badge rounded-pill text-bg-secondary-subtle border border-secondary-subtle ms-1">
                <?php echo h($r['category']); ?>
              </span>
            </td>
            <td>
              <strong><?php echo h($r['recipe_code']); ?></strong><br>
              <span class="text-muted"><?php echo h($r['recipe_name']); ?></span>
            </td>
            <td class="text-end"><?php echo number_format($yield); ?></td>
            <td class="text-end">KES <?php echo number_format($batch_cost, 2); ?></td>
            <td class="text-end">KES <?php echo number_format($cost_piece, 2); ?></td>
            <td class="text-end">KES <?php echo number_format($sp, 2); ?></td>
            <td class="text-end <?php echo $margin >= 0 ? 'text-success' : 'text-danger'; ?>">
              KES <?php echo number_format($margin, 2); ?>
            </td>
            <td class="text-end <?php echo $margin >= 0 ? 'text-success' : 'text-danger'; ?>">
              <?php echo number_format($margin_pct, 1); ?>%
            </td>
          </tr>
        <?php endwhile; ?>
        <?php if ($recipes->num_rows === 0): ?>
          <tr><td colspan="8" class="text-center text-muted py-3">
            No recipes yet. Create products, raw materials, then recipes & ingredients.
          </td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>
<?php require_once __DIR__ . "/../../includes/footer.php"; ?>
